const EXPORTED_SYMBOLS = ["module"];

const BRANDING_PROVIDER_ID = "ru.yandex.custombar.branding";
const BRANDING_SERVICE_NAME = "package";
const BRANDING_TOPIC_UPDATED = "package updated";

const BRANDING_DEFAULT_ID = "undefined";
const BRANDING_PATH_PREFIX = "/branding";
const BRANDING_MAP_FILE = "map.json";
const BRANDING_FILE_EXT = ".json";

var brandingCache = {};
var mappingCache = null;

var BRANDING_CHANGED_TOPIC = "branding-changed-" + (function (length) {
	var code,
	    result = Array(length);
	
	for (var i = 0; i < length; ++i) {
		code = 48 + Math.floor(Math.random() * 62);
		if (code > 0x39) code += 7;
		if (code > 0x5A) code += 6;
		result[i] = String.fromCharCode(code);
	}
	
	return result.join("");
}(32));

const nsObserverService = Components.classes["@mozilla.org/observer-service;1"];
const nsIObserverService = Components.interfaces.nsIObserverService;

var OS = nsObserverService.getService(nsIObserverService);

function module(application) {
	var log = function (x) {
		application.log("branding: " + x);
	};
	
	const barAPI = application.barAPI;
	const Package = barAPI.Package;
	
	var brandingService;
	
	function getBrandId() {
		var retVal = BRANDING_DEFAULT_ID;
		
		try {
			retVal = String(brandingService.getBrandID());
		} catch (e) {
			Components.utils.reportError(e);
		}
		
		return retVal;
	}
	
	function parseJSONFile(packagePath) {
		var path = Package.resolvePath(packagePath);
		var fileChannel = Package.getFileInputChannel(path);
		var data = barAPI.StrUtils.readStringFromStream(fileChannel.contentStream);
		return JSON.parse(data);
	}
	
	/**
	 * Возвращает объект-хранилище для бренда
	 * @param {String} brandKey имя бренда
	 * @param {String} filename имя файла без расширения
	 * @return {Object}
	 */
	function getBranding(brandKey, fileName) {
		var brandingConfigObject;
		var cacheKey = fileName.toLowerCase() + "|" + brandKey;
		
		if (!mappingCache) {
			try {
				mappingCache = parseJSONFile([BRANDING_PATH_PREFIX, BRANDING_MAP_FILE].join("/"));
			} catch (e) {
				mappingCache = {};
			}
		} 
		
		var mappedBrandKey = brandKey;
		
		if (mappingCache.hasOwnProperty(brandKey)) {
			mappedBrandKey = mappingCache[brandKey];
		}

		if (brandingCache.hasOwnProperty(cacheKey)) {
			brandingConfigObject = brandingCache[cacheKey];
		} else {
			try {
				brandingConfigObject = parseJSONFile([BRANDING_PATH_PREFIX, mappedBrandKey, fileName + BRANDING_FILE_EXT].join("/"));
			} catch (e) {
				// Components.utils.reportError(e);
				try {
					var defaultMappedBrandId = BRANDING_DEFAULT_ID;
					
					if (mappingCache.hasOwnProperty(BRANDING_DEFAULT_ID)) {
						defaultMappedBrandId = mappingCache[BRANDING_DEFAULT_ID];
					}
					
					brandingConfigObject = parseJSONFile([BRANDING_PATH_PREFIX, defaultMappedBrandId, fileName + BRANDING_FILE_EXT].join("/"));
				} catch (e) {
					Components.utils.reportError(e);
					brandingConfigObject = null;
				}
			}

			brandingCache[cacheKey] = brandingConfigObject;
		}
		
		return brandingConfigObject;
	}
	
	var brandingServiceListener = {
		observeServiceEvent: function (providerID, serviceName, topic, data) {
			switch (topic) {
				case BRANDING_TOPIC_UPDATED:
					OS.notifyObservers(null, BRANDING_CHANGED_TOPIC, "");
				break;
			}
		}
	};
	
	var BrandingOptionsStub = function () {};
	
	BrandingOptionsStub.prototype = {
		getOptions: function () { return null; },
		release: function () {},
		onbrandingchange: function () {}
	};
	
	var barServicesAvailable = false;
	
	if (typeof barAPI.Services !== "undefined") {
		barServicesAvailable = true;
		brandingService = barAPI.Services.obtainService(BRANDING_PROVIDER_ID, BRANDING_SERVICE_NAME, brandingServiceListener);
	}
	
	/**
	 * @constructor
	 * @param {String} filename
	 */
	var BrandingOptions = function (filename) {
		var brandId = getBrandId();
		var _this = this;
		
		this.onbrandingchange = function () {};
		
		this.getOptions = function () {
			return getBranding(getBrandId(), filename);
		};
		
		this.release = function () {
			release.call(this);
		};
		
		var finalizeTopic = "facebook";
		var finalizeData = "finalize";
		
		var brandingObserver = {
			observe: function (s, t, d) {
				_this.onbrandingchange();
			}
		};
		
		var finalizeObserver = {
			observe: function (s, t, d) {
				if (d == finalizeData) {
					log("finalize");
					release.call(_this);
				}
			}
		};
		
		function init() {
			OS.addObserver(brandingObserver, BRANDING_CHANGED_TOPIC, false);
			OS.addObserver(finalizeObserver, finalizeTopic, false);
		}
		
		function release() {
			OS.removeObserver(brandingObserver, BRANDING_CHANGED_TOPIC);
			OS.removeObserver(finalizeObserver, finalizeTopic);
		}
		
		init.call(this);
	}
	
	if (barServicesAvailable) {
		return BrandingOptions;
	} else {
		return BrandingOptionsStub;
	}
}