EXPORTED_SYMBOLS = ["module"];

var observerService = Components.classes["@mozilla.org/observer-service;1"].getService(Components.interfaces.nsIObserverService);
var timerFactory = Components.classes["@mozilla.org/timer;1"];

var module = function (application) {
	var settingsAPI = application.barAPI.Settings;
	
	var log = function (text) {
		application.log("counters: " + text);
	};
	
	function min2ts(x) {
		return x * 60 * 1000;
	}
	
	application.setCounterData = function (data) {
		application.counterData = data;
		application.notify("counter-data-changed");
	};
	
	application.changeCounterData = function (diff) {
		application.counterData = application.utils.mix(application.counterData, diff);
		application.notify("counter-data-changed");
	};

	
	var c = application.counters = {
		_updateInterval: Infinity, 
		_failUpdateInterval: 5 * 1000,
		_failNetworkDelay: 5 * 1000,
		_failCount: 0,
		_maxFails: 2,
		_lastUpdatedTimestamp: 0,
		_lastUpdatedError: 0,
		_updating: false,
		
		dataUpdateTimer: Components.classes["@mozilla.org/timer;1"].createInstance(Components.interfaces.nsITimer),
		
		setUpdateInterval: function (interval) {
			c._updateInterval = interval;
		},
		
		update: function () {
			log("update");
			c._updating = true;
			
			var counterData = {};
			
			var queries = {
				"me": {
					query: "SELECT uid, name FROM user WHERE uid = me()",
					parser: function (fql_result_set) {
						var user = fql_result_set[0];
						return {
							loggedIn: true,
							user: user
						}
					}
				},
				
				"notif": {
					query: "SELECT title_text FROM notification WHERE recipient_id = me() AND is_unread = 1 LIMIT 101",
					parser: function (fql_result_set) {
						return {
							notifications_count: fql_result_set.length,
							notifications: fql_result_set
						}
					}
				},
				
				"friendrequests": {
					query: "SELECT uid_from FROM friend_request WHERE uid_to = me() LIMIT 101",
					parser: function (fql_result_set) {
						return {
							friend_requests_count: fql_result_set.length,
							friend_requests: fql_result_set
						}
					}
				},
				
				"usernameresolver": {
					query: "SELECT uid, name FROM user WHERE " +
							"uid IN (SELECT uid_from FROM #friendrequests) " +
							"OR uid IN (SELECT snippet_author FROM #messages)",
					
					parser: function (fql_result_set) {
						return {
							users: fql_result_set
						}
					}
				},
				
				"messages_count": {
					query: "SELECT unread_count FROM mailbox_folder WHERE folder_id = 0",
					parser: function (fql_result_set) {
						var result = fql_result_set[0];
						return {
							messages_count: result.unread_count
						}
					}
				},
				
				"messages": {
					query: "SELECT thread_id, snippet_author, subject FROM thread WHERE folder_id = 0 AND unread != 0 LIMIT 10",
					parser: function (fql_result_set) {
						return {
							messages: fql_result_set
						}
					}
				}
			}
			
			function parseResult (res) {
				var counterData = {};
				
				for (var i = 0, l = res.length; i < l; ++i) {
					var response = res[i];
					var name = response.name;
					if (name in queries) {
						var result_set = response.fql_result_set;
						if (result_set) {
							counterData = application.utils.mix(counterData, queries[name].parser(result_set));
						}
					}
				}
				
				log("counterData = " + JSON.stringify(counterData));
				application.setCounterData(counterData);
			}
			
			var abortTimer = timerFactory.createInstance(Components.interfaces.nsITimer);
			
			var cb = new application.utils.CallbackObject(
				function (result) {
					log(JSON.stringify(result));
					abortTimer.cancel();
					c._updating = false;
					c._failCount = 0;
					c._lastUpdatedError = 0;
					c._lastUpdatedTimestamp = +new Date;
					parseResult(result);
				},
				
				function (error) {
					log("FQL error, error = " + error);
					abortTimer.cancel();
					c._updating = false;
					++c._failCount;
					c._lastUpdatedError = -1;
					if (c._failCount >= c._maxFails) {
						c._lastUpdatedTimestamp = +new Date;
						c._failCount = 0;
						if (error == "PARSE_ERROR") { // strange answer, no network
							var counterData = {};
							
							if (application.FB._credentials.access_token) {
								counterData.loggedIn = true; // assuming token is valid
								counterData.user = application.counterData.user;
								counterData.failState = true;
							}
							
							application.setCounterData(counterData);
						}
					}
				}
			)
			
			var q = {};
			
			for (var i in queries) {
				q[i] = queries[i].query;
			}
			
			var watcher = application.FB.execFQL(cb, q);
			
			abortTimer.initWithCallback(
				watcher.abort,
				c._failNetworkDelay,
				Components.interfaces.nsITimer.TYPE_ONE_SHOT
			);
		},
		
		observe: function (subj, topic, data) {
			switch (topic) {
				case "settings-changed":
					try {
						data = JSON.parse(data);
					} catch (e) {
						log("observe data error, data = " + data);
						return;
					}
					
					var field = "update-interval";
					
					if (field in data) {
						var updateInterval = data[field];
						c.setUpdateInterval(min2ts(updateInterval));
					}
					break;
					
				case "facebook":
					if (data == "finalize") {
						c.finalize();
					}
					break;
			}
		},
		
		_updateCheck: function () {
			var now = new Date;
			
			if (c._updating) {
				return;
			}
			
			if (!application.FB._credentials.access_token) {
				return;
			}
			
			var nextRequestInterval = c._lastUpdatedError ? c._failUpdateInterval : c._updateInterval;
			if (c._lastUpdatedTimestamp + nextRequestInterval < +now) {
				c.update();
			}
		},
		
		start: function () {
			var checkInterval = 1000;
			c.setUpdateInterval(min2ts(settingsAPI.getValue("update-interval")));
			c.update();
			c.dataUpdateTimer.initWithCallback(c._updateCheck, checkInterval, Components.interfaces.nsITimer.TYPE_REPEATING_SLACK);
			observerService.addObserver(c, "settings-changed", false);
			observerService.addObserver(c, "facebook", false);
		},
		
		finalize: function () {
			observerService.removeObserver(c, "facebook");
			observerService.removeObserver(c, "settings-changed");
			c.dataUpdateTimer.cancel();
		}
	}
}