"use strict";

var EXPORTED_SYMBOLS = ["module"];

var nsXMLHttpRequest = Components.classes["@mozilla.org/xmlextras/xmlhttprequest;1"];
var nsIXMLHttpRequest = Components.interfaces.nsIXMLHttpRequest;

function module(proxy) {
	var Microbrowser = proxy.module("UI.OAuthBrowser");
	var utils = proxy.module("utils.common");
			
	var APIError = function (code) {
		this.code = code;
	};
	
	APIError.messages = {
		1: "Unknown error: Please resubmit the request.",
		2: "Unknown method called.",
		3: "Method is deprecated.",
		100: "One of the parameters specified is missing or invalid.",
		102: "User authorization failed: the session key or uid is incorrect.",
		103: "Application lookup failed: the application id is not correct.",
		104: "Incorrect signature.",
		105: "Application is not installed for this user.",
		200: "Permission error: the application does not have permission to perform this action"
	};
	
	APIError.prototype = {
		constructor: APIError,
		getMessage: function (code) {
			var defaultMessage = "Unknown error";
			var message = APIError.messages[code] || defaultMessage;
			return message;
		}
	};
	
	var APIClient = function (appCredentials, userCredentials) {
		this.appCredentials = appCredentials;
		this.userCredentials = userCredentials || {};
		this.connected = false;
		
		if (this.userCredentials.access_token) {
			this.connected = true;
			this.startAccessTokenUpdates();
		}
		
		this.ondataready = function () {}
	};
	
	APIClient.OAUTH_CONNECT_URL = "https://connect.mail.ru/oauth/authorize";
	APIClient.OAUTH_TOKEN_UPDATE_URL = "https://appsmail.ru/oauth/token";
	APIClient.OAUTH_CALLBACK_URL = "http://connect.mail.ru/oauth/success.html";
	APIClient.API_URL = "http://www.appsmail.ru/platform/api";
	APIClient.LOGOUT_URL = "http://e.mail.ru/cgi-bin/logout";
	APIClient.LOGOUT_URL_REGEX = /^https?:\/\/([0-9a-z_-]+\.)?mail\.ru\/(cgi-bin\/logout|logout\.p?html)/i;
	APIClient.LOGIN_URL_REGEX = /^https?:\/\/auth\.mail\.ru\/cgi-bin\/auth/i;

	
	APIClient.prototype = {

        finalize: function () {
            this.cancelAccessTokenUpdate();
        },
		
		loginByCookie: function(callbacks) {
			
			var url = APIClient.OAUTH_CONNECT_URL + "?" + utils.obj2UrlParams({
					client_id: this.appCredentials.client_id,
					response_type: "token",
					redirect_uri: APIClient.OAUTH_CALLBACK_URL
				});
			
			var _this = this;
			
			//proxy.logger.debug('loginByCookie, url=' + url);
			
			var request = nsXMLHttpRequest.createInstance(nsIXMLHttpRequest);
            //request.mozBackgroundRequest = true;
			request.open('GET', url);
			

			request.onreadystatechange = function (event) {
				var target = event.target;
				var state = target.readyState;
				if (state == 4) {
					var status = target.status;
					if (status >= 200 && status < 400) {
						var url = request.channel.URI.spec;
						//proxy.logger.debug('loginByCookie response, url=' + url);
						if ( ! utils.callbacksMatch(APIClient.OAUTH_CALLBACK_URL, url) ) {
							// it is not a callback url
							return;
						}
						//proxy.logger.debug('loginByCookie auth!!!');
						_this.handleAuth(url, callbacks);
					}
				}
			};
			
			
			request.send();
		},

		connectAsync: function (callbacks) {
			var _this = this;
			
			var features = "centerscreen, width=600, height=440, resizable";
			var args = {
				url: APIClient.OAUTH_CONNECT_URL + "?" + utils.obj2UrlParams({
					client_id: this.appCredentials.client_id,
					response_type: "token",
					redirect_uri: APIClient.OAUTH_CALLBACK_URL
				})
			};
			
			var microbrowser = new Microbrowser();
			
			microbrowser.onstatechange = function (progress, request) {
                var url = request.name;

                if ( ! utils.callbacksMatch(APIClient.OAUTH_CALLBACK_URL, url) ) {
                    // it is not a callback url
                    return;
                }
                if (microbrowser.window.closed) {
                    return;
                }
                
                request.cancel(1);
                microbrowser.close();
				
				_this.handleAuth(url, callbacks);
			};
			
			microbrowser.onclose = function (event) {
				callbacks.error("CANCELLED");
			};
			
			microbrowser.open(features, args);
		},
		
		handleAuth: function(url, callbacks) {
			var _this = this;
			
			url = url.replace('?', '#');
			var credentialsText = url.split("#")[1];
			var credentials = utils.urlParams2Obj(credentialsText);
			proxy.logger.trace("Mailru credentials = " + JSON.stringify(credentials));

			// user cancels authorization request or some other error has occurred
			if('undefined' != typeof credentials.error) {
				_this.connected = false;
				return callbacks.error("RESPONSE_ERROR");
			}

			for (var key in credentials) {
				if (credentials.hasOwnProperty(key)) {
					_this.userCredentials[key] = credentials[key];
				}
			}

			if (credentials) {
				_this.connected = true;

				_this.userCredentials.expiration_time = _this._getCurrentTime() + Number(credentials.expires_in);
				_this.startAccessTokenUpdates();

				return _this.getMailboxAddress({success: function (mailbox) {
					_this.userCredentials.mailbox = mailbox;
					return callbacks.success();
				}, error: function () {
					_this.userCredentials.mailbox = "";
					return callbacks.success();
				}});
			} else {
				_this.connected = false;
				return callbacks.error("RESPONSE_ERROR");
			}
		},

        startAccessTokenUpdates: function () {
            let expirationTime = this.userCredentials.expiration_time;
            let currentTime = this._getCurrentTime();

            let accessTokenIsExpired = ( expirationTime < currentTime);
            
            if ( accessTokenIsExpired ) {
                this.updateAccessToken();
            }
            else {
                let delay = expirationTime - currentTime;
                this.scheduleNextAccessTokenUpdate(delay);
            }
        },

        _updatesTimer: null,
        scheduleNextAccessTokenUpdate: function (delayInSeconds) {
            this.cancelAccessTokenUpdate();

            // yes, I have to create new instance every time
            this._updatesTimer = Components.classes["@mozilla.org/timer;1"]
                .createInstance(Components.interfaces.nsITimer);

            var _this = this;
            this._updatesTimer.init(
                {
                    observe: function () {
                        _this.updateAccessToken();
                    }
                },
                delayInSeconds * 1000,
                Components.interfaces.nsITimer.TYPE_ONE_SHOT
            );
            proxy.logger.debug("Access token will be updated in " + delayInSeconds + " seconds.");
        },

        cancelAccessTokenUpdate: function () {
            if (null !== this._updatesTimer) {
                this._updatesTimer.cancel();
            }
        },


        updateAccessToken: function () {
            var _this = this;
            this.getAccessToken({
                success: function (response) {
                    let accessToken = response.access_token;
                    if('undefined' == typeof accessToken) {
                        return;
                    }
                    _this.userCredentials.access_token = accessToken;
                    proxy.logger.trace("Access token is updated. New one is '" + accessToken + "'.");

                    let expiresIn = response.expires_in;
                    _this.userCredentials.expires_in = expiresIn;
                    _this.userCredentials.expiration_time = expiresIn + _this._getCurrentTime();

                    let delay = _this.userCredentials.expiration_time;
                    _this.scheduleNextAccessTokenUpdate(delay);
                },
                error: function () {
                    proxy.logger.trace('An error occurred while updating access token.');
                }
            });
        },

        getAccessToken: function (callbacks) {
            var _this = this;

            let url = APIClient.OAUTH_TOKEN_UPDATE_URL + "?" + utils.obj2UrlParams(
                {
                    client_id: this.appCredentials.client_id,
					grant_type: 'refresh_token',
                    refresh_token: this.userCredentials.refresh_token,
                    client_secret: this.appCredentials.secret_key
                }
            );
			var request = nsXMLHttpRequest.createInstance(nsIXMLHttpRequest);
			request.mozBackgroundRequest = true;

			request.open('POST', url);

			request.onreadystatechange = function (event) {
				var target = event.target;
				var state = target.readyState;

				if (state == 4) {
					var status = target.status;
					switch (true) {
						case status >= 200 && status < 300:
							var response = null;

							try {
								response = JSON.parse(target.responseText);
							} catch (e) {
								proxy.logger.trace("Error parsing response. See next message for details:");
								proxy.logger.error(e);
                                return callbacks.error("PARSE_ERROR");
							}

							if ('object' != typeof response || null === response) {
                                return callbacks.error("NO_RESPONSE");
                            }

                            callbacks.success(response);
                            break;
						case status == 0:
							return callbacks.error("ABORTED");
						break;
						default:
							return callbacks.error("ERROR");
					}
				}
			};

			request.send(null);
        },

        _getCurrentTime: function () {
            return Math.floor((new Date()).getTime() / 1000);
        },
		
		getMailboxAddress: function (callbacks) {
			return this.getUserInfoAsync({success: function (data) {
				var link = data.link;
				
				proxy.logger.trace("Parsing mailbox from \"%s\"".replace("%s", link));
				
				var exRx = /\/\/my\.mail\.ru\/(.*?)\/(.*?)\//;
				var parts = exRx.exec(link);
				if (parts && parts.length == 3) {
					var domain = parts[1];
					var user = parts[2];
					var address = user + "@" + domain + ".ru";
					proxy.logger.trace("Parsing mailbox result is \"%s\"".replace("%s", address));
					return callbacks.success(address);
				} else {
					proxy.logger.trace("Parsing mailbox failed");
					return callbacks.error("PARSE_ERROR");
				}
			}, error: callbacks.error});
		},
		
		getUserInfoAsync: function (callbacks) {
			return this.queryAsync({
				success: function (data) {
					return callbacks.success(data[0]);
				},
				error: callbacks.error
			}, "GET", "users.getInfo");
		},
		
		getUnreadCountAsync: function (callbacks) {
			return this.queryAsync({
				success: function (data) {
					return callbacks.success(data.count);
				},
				error: callbacks.error
			}, "GET", "mail.getUnreadCount");
		},

        logout: function () {
			var request = nsXMLHttpRequest.createInstance(nsIXMLHttpRequest);
            request.open('GET', APIClient.LOGOUT_URL);
			request.send();

            this.cancelAccessTokenUpdate();
        },

		queryAsync: function (callbacks, httpMethod, apiMethod, queryParams) {
			var logPrefix = "Mailru APIClient queryAsync %s".replace("%s", apiMethod);
			proxy.logger.trace(logPrefix);
			
			if (!this.connected) {
				callbacks.error("NOT_CONNECTED");
				return;
			}
			
			var params = {
				session_key: this.userCredentials.access_token,
				method: apiMethod,
				app_id: this.appCredentials.client_id,
				secure: 0,
				format: "json"
			};
			
			params = utils.mix(params, queryParams);
			
			var paramsSigBasePart = utils.obj2UrlParams(params).replace(/&/g, "");
			
			var sigBase = params.secure
				? [paramsSigBasePart, this.appCredentials.secret_key].join("")
				: [this.userCredentials.x_mailru_vid, paramsSigBasePart, this.appCredentials.private_key].join("");
			
			params.sig = utils.digest("md5", sigBase);
			
			var queryUrl = [APIClient.API_URL, utils.obj2UrlParams(params)].join("?");
			
			var request = nsXMLHttpRequest.createInstance(nsIXMLHttpRequest);
			request.mozBackgroundRequest = true;
			
			request.open(httpMethod, queryUrl);
			
			request.onreadystatechange = function (event) {
				var target = event.target;
				var state = target.readyState;

				if (state == 4) {
					var status = target.status;
					switch (true) {
						case status >= 200 && status < 300:
							var response = null;
							
							try {
								response = JSON.parse(target.responseText);
							} catch (e) {
								proxy.logger.trace("Error parsing response. See next message for details:");
								proxy.logger.error(e);
							}
							
							if (response) {
								return callbacks.success(response);
							} else {
								return callbacks.error("PARSE_ERROR");
							}
						break
						case status == 0:
							return callbacks.error("ABORTED");
						break;
						default:
							return callbacks.error("ERROR");
					}
				}
			};
			
			request.send("");
			
			return {
				abort: function () {
					request.abort();
				}
			};
		}
	};
	
	var mailru_ns = {
		APIClient: APIClient
	};
	
	return mailru_ns;
}