"use strict";

var EXPORTED_SYMBOLS = ["module"];

var NETWORK_TIMEOUT = 5000;

var nsTimer = Components.classes["@mozilla.org/timer;1"];
var nsITimer = Components.interfaces.nsITimer;

function module (proxy) {
	var UpdateTimer = proxy.module("app.UpdateTimer");
	var T = proxy.module("sklib.Template");
	var settings = proxy.module("bar.settings");
	
	var navigatePages = {
		inbox: "http://e.mail.ru/cgi-bin/msglist?folder=0",
		read: "http://e.mail.ru/cgi-bin/msglist?folder=0",
		compose: "http://e.mail.ru/cgi-bin/sentmsg?compose",
		addressbook: "http://e.mail.ru/cgi-bin/addressbook"
	};
	
	var MailruActionsWrapper = function MailruActionsWrapper (apiClient, storage) {
		
		var MailruActions = function () {
			var _this = this;
			var updateIntervalKey = "update-interval";
			
			this.uiManager = null;
			this.updateTimer = new UpdateTimer(settings.getTimerValueAsMinutes(updateIntervalKey));
			
			settings.addListener(function (topic, key, value) {
				if (key == updateIntervalKey) {
					_this.updateTimer.setUpdateInterval(settings.getTimerValueAsMinutes(updateIntervalKey));
				}
			});
			
			this.updateTimer.ontimer = function (callbacks) {
				_this._update(callbacks);
			};
			
			this.updateTimer.start(true);
		};
		
		MailruActions.prototype = {
			constructor: MailruActionsWrapper,
			
			setUIManager: function (manager) {
				this.uiManager = manager;
			},
			
			setStorageData: function (key, value) {
				storage[key] = value;
				
				if (this.uiManager) {
					this.uiManager.display(storage);
				}
			},
			
			execute: function (action) {
				proxy.logger.trace(T("Execute mailru action $0", [action]));
				
				var actionArray = action.split("-");
				var actionName = actionArray.shift();
				
				var command = this[actionName];
				
				if (command) {
					command.apply(this, actionArray);
				}
			},
			
			navigate: function (page) {
				proxy.logger.trace(T("Navigate to page $0", [page]));
				
				var url = navigatePages[page];
				if (page) {
					proxy.api.Controls.navigateBrowser({
						url: url,
						target: "new tab"
					});
				}
			},
			
			_update: function (callbacks) {
				var _this = this;
				
				var failTimer = nsTimer.createInstance(nsITimer);
				var watcher;
				
				proxy.logger.trace("Init watchdog timer");
				
				failTimer.initWithCallback(function () {
					proxy.logger.trace("Network timeout");
					
					_this.setStorageData("unread", undefined);
					_this.setStorageData("error", "NETWORK_TIMEOUT");
					
					watcher.abort();
					
					callbacks.error();
				}, NETWORK_TIMEOUT, nsITimer.TYPE_ONE_SHOT);
				
				watcher = apiClient.getUnreadCountAsync({
					success: function (data) {
						proxy.logger.trace(T("Got unerad count = $0. Cancel watchdog timer", [data]));
						
						failTimer.cancel();
						_this.setStorageData("unread", data);
						_this.setStorageData("error", undefined);
						
						callbacks.success();
					},
					error: function (error) {
						proxy.logger.trace(T("Get unerad count error $0. Cancel watchdog timer", [error]));
						
						failTimer.cancel();
						_this.setStorageData("unread", undefined);
						_this.setStorageData("error", error);
						
						callbacks.error();
					}
				});
			},
			
			update: function (callbacks) {
				proxy.logger.trace("Update command");
				
				this.updateTimer.update();
			},
			
			auth: function (successNavigateTo) {
				proxy.logger.trace("Auth command");
				
				var _this = this;
				
				apiClient.connectAsync({success: function () {
					proxy.logger.trace("Successful connect");
					
					_this.setStorageData("mailbox", apiClient.userCredentials.mailbox);
					_this.setStorageData("auth", true);
					_this.update();
					
					if (successNavigateTo) {
						_this.navigate(successNavigateTo);
					}
				}, error: function (error) {
					proxy.logger.error("Connect error " + error);
					
					_this.setStorageData("auth", false);
					_this.setStorageData("mailbox", undefined);
				}});
				
				this.updateTimer.start(true);
			},
			
			authByCookie: function () {
				proxy.logger.debug("authByCookie command");
				
				var _this = this;
				
				apiClient.loginByCookie({success: function () {
					proxy.logger.trace("Successful connect");
					
					_this.setStorageData("mailbox", apiClient.userCredentials.mailbox);
					_this.setStorageData("auth", true);
					_this.update();
					
				}, error: function (error) {
					proxy.logger.error("Connect error " + error);
					
					_this.setStorageData("auth", false);
					_this.setStorageData("mailbox", undefined);
				}});
				
				this.updateTimer.start(true);
			},
			
			logout: function (passiveLogout) {
				for (var i in storage) {
					delete storage[i]; 
				}
				
				if (this.uiManager) {
					this.uiManager.display(storage);
				}
				
				this.updateTimer.stop();
                
                if (!passiveLogout)
                    apiClient.logout();
			},
			
			options: function () {
				var wiid = this.uiManager && this.uiManager.wiid || null;
				
				proxy.logger.trace(T("Open options WIID = $0", [wiid]));
				proxy.api.Controls.openSettingsDialog(null, wiid);
			}
		};
		
		return new MailruActions;
	};
	
	return MailruActionsWrapper;
};
