Y.UI.TwitLinks = (function () {
	var TwitEntity = {};

	function fixHTML(aStr) {
		return aStr.replace("&amp;lt;", "&lt;", "g")
			.replace("&amp;amp;", "&amp;", "g")
			.replace("&amp;quot;", "&quot;", "g")
	.replace("&amp;gt;", "&gt;", "g");
	}
	
	// TODO ref this
	var TwitFixer = {
		/**
		 * Получает положение начала сущности из элемента
		 * @param {Element} aElement
		 * @returns Number
		 */
		getStartFromElement: function (aElement) {
			return Number(aElement.getAttribute("start"));
		},
		
		/**
		 * Получает положение конца сущности из элемента
		 * @param {Element} aElement
		 * @returns Number
		 */
		getEndFromElement: function (aElement) {
			return Number(aElement.getAttribute("end"));
		}
	};
	
	var TwitEntityPrototype = 
		/**
		 * @constructor
		 * @param {Number} aStart Начало
		 * @param {Number} aEnd Конец
		 */
		function (aStart, aEnd) {
			this.start = aStart;
			this.end = aEnd;
		};
	
	TwitEntityPrototype.prototype = {
		/**
		 * Получает положение начала сущности из элемента
		 * @param {Element} aElement
		 * @returns Number
		 */
		getStartFromElement: function (aElement) {
			return Number(aElement.getAttribute("start"));
		},
		
		/**
		 * Получает положение конца сущности из элемента
		 * @param {Element} aElement
		 * @returns Number
		 */
		getEndFromElement: function (aElement) {
			return Number(aElement.getAttribute("end"));
		}
	};
	
	/**
	 * Ссылка внутри текста твита
	 * 
	 * @constructor
	 * @param aStart Положение начала
	 * @param aEnd Положение конца
	 * @param aUrl Адрес ссылки
	 * @param aText Текст внутри ссылки
	 * @param aTitle Тайтл (заголовок) ссылки
	 */
	TwitEntity.url = function (aStart, aEnd, aUrl, aText, aTitle) {
		TwitEntityPrototype.apply(this, arguments);
		
		this.url = aUrl || "";
		this.text = aText || this.url;
		this.title = aTitle || this.url;
	};
	
	TwitEntity.url.prototype = new TwitEntityPrototype;
	TwitEntity.url.prototype.toString = function () {
		return '<a href="javascript:twitlink.open(\'$url\')" title="$title">$text</a>'.displayAsEscapedHTML(this);
	};
	
	/**
	 * Создает сущность из элемента &lt;url&gt;
	 * @param {Element} element
	 * @returns TwitEntity.url
	 */
	TwitEntity.url.createFromElement = function (element) {
		var start = 0,
			end = 0,
			url = "",
			text = "",
			title = "";
		
		start = TwitFixer.getStartFromElement(element);
		end = TwitFixer.getEndFromElement(element);
		
		var childHandlers = {
			url: function (element) {
				url = element.innerHTML || element.textContent;
			},
			
			display_url: function (element) {
				text = element.innerHTML || element.textContent;
			},
			
			expanded_url: function (element) {
				title = element.innerHTML || element.textContent;
			}
		};
		
		var childs = element.children;
		
		for (var i = 0, child; child = childs[i]; ++i) {
			var tagName = child.tagName.toLowerCase();
			var handler = childHandlers[tagName];
			
			handler && handler(child);
		}
		
		return new TwitEntity.url(start, end, url, text, title);
	};
	
	/**
	 * Хештег внутри текста твита
	 * @param aTag
	 * @returns {String}
	 */
	TwitEntity.hashtag = function (aStart, aEnd, aTag) {
		TwitEntityPrototype.apply(this, arguments);
		this.tag = aTag || "";
	};
	
	TwitEntity.hashtag.prototype = new TwitEntityPrototype;
	TwitEntity.hashtag.prototype.toString = function () {
		return '<a href="javascript:twitlink.search(\'$tag\')">$tag</a>'.displayAsEscapedHTML(this);
	};
	
	TwitEntity.hashtag.createFromElement = function (element) {
		var text = "";
		
		start = TwitFixer.getStartFromElement(element);
		end = TwitFixer.getEndFromElement(element);
		
		var childHandlers = {
			text: function (element) {
				text = "#" + (element.innerText || element.textContent);
			}
		};
		
		var childs = element.children;
		
		for (var i = 0, child; child = childs[i]; ++i) {
			var tagName = child.tagName.toLowerCase();
			var handler = childHandlers[tagName];
			
			handler && handler(child);
		}
		
		return new TwitEntity.hashtag(start, end, text);
	};

	TwitEntity.creative = TwitEntity.url;

	TwitEntity.user_mention = function (aStart, aEnd, aScreenName, aName) {
		TwitEntityPrototype.apply(this, arguments);
		this.screen_name = aScreenName || "";
		this.name = aName || "";
	};
	
	TwitEntity.user_mention.prototype = new TwitEntityPrototype;
	TwitEntity.user_mention.prototype.toString = function () {
		return '<a href="javascript:twitlink.info(\'$screen_name\')" title="$name">$screen_name</a>'.displayAsEscapedHTML(this);
	};
	
	TwitEntity.user_mention.createFromElement = function (element) {
		var start,
			end,
			screenName = "",
			name = "";
		
		start = TwitFixer.getStartFromElement(element);
		end = TwitFixer.getEndFromElement(element);
		
		var childHandlers = {
			name: function (element) {
				name = element.innerText || element.textContent;
			},
			screen_name: function (element) {
				screenName = "@" + (element.innerText || element.textContent);
			}
		};
		
		var childs = element.children;
		
		for (var i = 0, child; child = childs[i]; ++i) {
			var tagName = child.tagName.toLowerCase();
			var handler = childHandlers[tagName];
			
			handler && handler(child);
		}
		
		return new TwitEntity.user_mention(start, end, screenName, name);
	};
	
	function createFieldSorter(fieldname) {
		return function (a, b) {
			var x = a[fieldname],
				y = b[fieldname];
		
			return x < y ? -1 : x === y ? 0 : 1;
		};
	}
	
	var entitySorter = createFieldSorter("start");
	
	function createEntityListFromElement(element, html) {
		var retVal = [];
		var childs = element.getElementsByTagName("*");
		
		for (var i = 0, child; child = childs[i]; ++i) {
			var tagName = child.tagName.toLowerCase();
			
			if (child.getAttribute("start") !== null) {
				if (TwitEntity.hasOwnProperty(tagName)) {
					retVal.push(TwitEntity[tagName].createFromElement(child));
				}
			}
		}
		
		retVal.sort(entitySorter);
		
		return retVal;
	}
	
	function createEntityListFromText(text) {
		var retVal = [];
		
		var hashtagRx = /(?:^|[\s\W])(#[\u0041-\u005a\u0061-\u007a\u00c0-\u1fff\-\'\u2019_]+)/g;
		var mentionRx = /(?:^|[\s\W])(@[\w_]+)/g;
		var urlRx =     /(?:^|[\s\W])((?:https?|s?ftp):(?:\/\/)?[a-z0-9\.\-_]+(?:[\/\?\=\&_0-9\u0041-\u005a\u0061-\u007a\u00c0-\u1fff\u2019]+)?)/ig;
		var res = [];
		
		while (res = hashtagRx.exec(text)) {
			if (res.length > 1) {
				var match = res[0];
				var tag = res[1];
				var index = res.index;
				var length = tag.length;
				var delta = match.indexOf(tag);
				index = index + delta;
				retVal.push(new TwitEntity.hashtag(index, index + length, tag));
			}
		}

		while (res = mentionRx.exec(text)) {
			if (res.length > 1) {
				var match = res[0];
				var mention = res[1];
				var index = res.index;
				var length = mention.length;
				var delta = match.indexOf(mention);
				index = index + delta;
				retVal.push(new TwitEntity.user_mention(index, index + length, mention));
			}
		}

		while (res = urlRx.exec(text)) {
			if (res.length > 1) {
				var match = res[0];
				var url = res[1];
				var index = res.index;
				var length = url.length;
				var delta = match.indexOf(url);
				index = index + delta;
				retVal.push(new TwitEntity.url(index, index + length, url));
			}
		}
		
		retVal.sort(entitySorter);
		
		return retVal;
	}

	function Constructor (container) {
		var markAttributeName = "twitlink-processed";
		if (container.getAttribute(markAttributeName) == "true") {
			return;
		}
		
		var html = container.innerHTML.replace("&amp;", "&", "g"); // due twitter &amp;amp; bug 
		
		var id = container.getAttribute("id");
		
		if (id) {
			var entitiesId = id + "-entities";
			var entitiesElement = document.getElementById(entitiesId);
		} else {
			return;
		}
		
		var startIndex = 0;
		var pieces = []; // кусочки твита
		var entitiesList = [];

		if (entitiesElement) { // если энтити пришли от апи, то используем их
			entitiesList = createEntityListFromElement(entitiesElement, html);
		} else {               // если энтитей нет - колхозим сами
			entitiesList = createEntityListFromText(html);
		}
		
		for (var i = 0, entity; entity = entitiesList[i]; ++i) {
			pieces.push(html.substring(startIndex, entity.start));
			pieces.push(entity);
			startIndex = entity.end;
		}
		
		pieces.push(html.substring(startIndex)); // все что осталось после последней итерации
		
		html = pieces.join("").replace("&", "&amp;", "g"); // due twitter &amp;amp; bug 
		
		container.innerHTML = fixHTML(html); // this is SPARTAA!!!!111
		container.setAttribute(markAttributeName, "true");
	};
	
	Constructor.prototype = {
		constructor: Constructor
	};
	
	return Constructor;
})();