var EXPORTED_SYMBOLS = ["module"];

var nsTimer = Components.classes['@mozilla.org/timer;1'];
var	nsITimer = Components.interfaces.nsITimer;

function module(application) {
	function log(x) {
		return application.log("cache: " + x);
	}
	
	/**
	 * @constructor
	 * @class Cache
	 * @param {String} filename
	 */
	function Cache(filename) {
		var SO = application.importModule("StorableObject");
		
		/** @private */
		this._filename = filename;

		/** @private */
		this._itemLifeTime = 108e5; // 3h
		
		/** @private */
		this._saveDelay = 1e3; // 1s
		
		/** @private */
		this._items = [];
		
		/** @private */
		this._itemExpires = [];
		
		/** @private */
		this._storage = new SO();
		
		/** @private */
		this._saveTimer = nsTimer.createInstance(nsITimer);
		
		this._load();
	}
	
	/**
	 * @class Cache
	 */
	Cache.prototype = {
		constructor: Cache,
		
		push: function (value) {
			if (this._items.indexOf(value) !== -1) {
				return;
			}

			var now = +new Date;
			this._items.push(value);
			this._itemExpires.push(now + this._itemLifeTime);
			this._saveDelayed();
		},
		
		getAll: function () {
			this._shrinkCache();
			return this._items;
		},
		
		/**
		 * @private
		 * @return {boolean}
		 */
		_shrinkCache: function () {
			var now = new Date;
			var items = this._items;
			var newItems = [];
			var expires = this._itemExpires;
			var newExpires = [];
			var needShrink = false;
			var isShrinked = false;
			
			for (var i = 0, l = expires.length; i < l; ++i) {
				if (expires[i] < now) {
					needShrink = true;
					break;
				}
			}
			
			if (needShrink) {
				for (var i = 0, l = expires.length; i < l; ++i) {
					if (expires[i] > now) { // если запись еще валидная
						newExpires.push(expires[i]);
						newItems.push(items[i]);
					}
				}
				
				this._items = newItems;
				this._itemExpires = newExpires;
				
				isShrinked = true;
			}
			
			return isShrinked;
		},
		
		/**
		 * @private 
		 */
		_saveDelayed: function () {
			var _this = this;
			this._saveTimer.initWithCallback(function () {
				_this._save();
			}, this._saveDelay, nsITimer.TYPE_ONE_SHOT);
		},
		
		/** @private */
		_cancelSaveDelayed: function () {
			this._saveTimer.cancel();
		},
		
		/**
		 * @private
		 */
		_save: function () {
			this._cancelSaveDelayed();
			this._shrinkCache();
			
			var storage = this._storage;

			storage["items"] = this._items;
			storage["expires"] = this._itemExpires;
			storage.__save__(this._filename);
			
			delete storage["items"];
			delete storage["expires"];
		},
		
		/**
		 * @private
		 */
		_load: function () {
			this._cancelSaveDelayed();
			
			var storage = this._storage;
			
			storage.__load__(this._filename);
			this._items = storage["items"];
			this._itemExpires = storage["expires"];
			
			delete storage["items"];
			delete storage["expires"];
			
			if (!this._cacheIsValid()) {
				this._items = [];
				this._itemExpires = [];
			}
			
			this._shrinkCache();
		},
		
		/**
		 * @private
		 * @return {boolean}
		 */
		_cacheIsValid: function () {
			var isValid = false;
			
			try {
				isValid = this._items.length === this._itemExpires.length;
			} catch (e) {
				isValid = false;
			}
			
			return isValid;
		}
	};
	
	application.removedMessages = new Cache("removed-messages.json");

	return Cache;
}